using System;
using System.Collections.Generic;
using System.ComponentModel.DataAnnotations;
using System.Linq;
using System.Web.Mvc;
using HIPS.Web.Components.Collections;
using HIPS.Web.Model.AssistedRegistration;

namespace HIPS.Web.UI.ViewModels.AssistedRegistration
{
    /// <summary>
    ///     Represents demographic/identifying information for a person.
    /// </summary>
    public class PersonDemographicViewModel
    {
        #region Properties

        public string Ihi { get; set; }

        [Required]
        [Display(Name = "Family Name")]
        public string FamilyName { get; set; }

        [Required]
        [Display(Name = "Given Name")]
        public string GivenNames { get; set; }

        [Required]
        [Display(Name = "Date of Birth"), DisplayFormat(DataFormatString = "dd/MM/yyyy")]
        public DateTime? DateOfBirth { get; set; }

        [Required]
        [Display(Name = "Sex")]
        public string SexCode { get; set; }

        [Display(Name = "Aboriginal or Torres Strait Islander Origin")]
        public string IndigenousStatusCode { get; set; }

        [Display(Name = "Medicare Card Number")]
        public string MedicareNumber { get; set; }

        [Display(Name = "IRN")]
        public string MedicareNumberIrn { get; set; }

        [Display(Name = "DVA File Number")]
        public string DvaFileNumber { get; set; }

        #endregion

        #region Calculated Properties

        /// <summary>
        ///     Indicates this view model represents known patient data,
        ///     including a validated IHI.
        /// </summary>
        public bool IsKnownPatient
        {
            get { return !String.IsNullOrWhiteSpace(Ihi); }
        }

        #endregion

        #region View Logic Properties

        /// <summary>
        ///     If true, the DVA File Number is a valid identification option.
        /// </summary>
        public bool ShowDvaFileNumber { get; internal set; }

        /// <summary>
        ///     If true, Indigenous Status will be requested.
        /// </summary>
        public bool ShowIndigenousStatus { get; internal set; }

        #endregion

        #region Data Source Properties

        public IEnumerable<SelectListItem> Sexes { get; internal set; }

        public IEnumerable<SelectListItem> IndigenousStatuses { get; internal set; }

        #endregion

        public PersonDemographicViewModel()
        {
            ShowDvaFileNumber = true;
            ShowIndigenousStatus = true;
            Sexes = Enumerable.Empty<SelectListItem>();
            IndigenousStatuses = Enumerable.Empty<SelectListItem>();
        }

        #region Data Source Methods

        /// <summary>
        ///     Populates the data source of Indigenous Statuses for this view model and nested entities.
        /// </summary>
        /// <param name="indigenousStatuses">A data source of possible Indigenous Statuses.</param>
        public void SetIndigenousStatuses(List<IndigenousStatus> indigenousStatuses)
        {
            IndigenousStatuses = indigenousStatuses.ToSelectListItems(i => i.Code, i => i.Description);
        }

        /// <summary>
        ///     Populates the data source of Sexes for this view model and nested entities.
        /// </summary>
        /// <param name="sexes">A data source of possible Sexes.</param>
        public void SetSexes(List<Sex> sexes)
        {
            Sexes = sexes.ToSelectListItems(s => s.Code, s => s.Description);
        }

        public IndigenousStatus GetIndigenousStatus(List<IndigenousStatus> indigenousStatuses)
        {
            return indigenousStatuses.FirstOrDefault(i => i.Code == IndigenousStatusCode);
        }

        public Sex GetSex(List<Sex> sexes)
        {
            return sexes.FirstOrDefault(i => i.Code == SexCode);
        }

        #endregion

        #region Methods

        /// <summary>
        ///     Loads known Applicant data from the provided PatientWithoutPcehr model.
        /// </summary>
        /// <param name="patient">A known PatientWithoutPcehr receord.</param>
        public void LoadPatient(ValidatedPatient patient)
        {
            if (patient == null)
            {
                throw new ArgumentNullException("patient", "Cannot load from a null patient.");
            }

            FamilyName = patient.RegisteredDemographics.FamilyName;
            GivenNames = patient.RegisteredDemographics.GivenName;
            MedicareNumber = patient.RegisteredDemographics.MedicareNumber;
            MedicareNumberIrn = patient.RegisteredDemographics.MedicareIrn;
            DvaFileNumber = patient.RegisteredDemographics.DvaFileNumber;
            DateOfBirth = patient.RegisteredDemographics.DateOfBirth;
            Ihi = patient.VerifiedIhi.Ihi;
            SexCode = patient.RegisteredDemographics.Sex.Code;
        }

        public PersonDemographic ToPersonDemographicModel(List<Sex> sexes)
        {
            Sex personSex = sexes.FirstOrDefault(s => s.Code == SexCode);
            return new PersonDemographic(FamilyName, GivenNames, DateOfBirth, personSex, MedicareNumber, MedicareNumberIrn, DvaFileNumber);
        }


        #endregion
    }
}